﻿using System.Collections.Generic;
using System.Collections.Specialized;
using System.Globalization;
using System.Linq;
using System.Web;
using Newtonsoft.Json;
using Sitecore;
using Sitecore.Data;
using Sitecore.Data.Managers;
using Sitecore.Globalization;
using Smartling.Connector.Extensions;
using Smartling.Connector.Model;
using Smartling.Connector.Security;
using Smartling.Connector.Translation;
using Smartling.Connector.Translation.Status;

namespace Smartling.Connector.sitecore_modules.Shell.Smartling.Connector
{
  /// <summary>
  /// Summary description for Items
  /// </summary>
  public class Items : IHttpHandler
  {
    private const string JsonContentType = "application/json";
    private const string PercentCompleteTemplate = "<span class='smartlingCompletion' itemId='{0}' targetLanguage='{1}'>{2}%</span>";
    private const string PercentCompleteMediaTemplate = "<span>{0}%</span>";
    private const string ContentFileType = "Content";
    private const string MediaFileType = "Media File";
    private const string LongDateFormat = "dd MMM, yyyy HH:mm:ss tt";
    private const string ShortDateFormat = "dd MMM, yyyy";
    private const string LengthParamName = "length";
    private const string StartParamName = "start";
    private const string DrawParamName = "draw";
    private const string OrderByParamName = "orderby";
    private const string OrderDirParamName = "orderdir";
    private const string PathColumnIndex = "1";
    private const string SubmittedColumnIndex = "9";
    private const string DateFormatString = "<span style=\"display:none\">{0}</span><span title='{1}'>{2}</span>";
    private readonly Dictionary<string, string>  languages = new Dictionary<string, string>();
    private static readonly Dictionary<string, string> statusTitles = new Dictionary<string, string>()
    {
      { "Waiting", "New" },
      { "Pushed", "In Progress" },
      { "Failed", "Failed" },
      { "Pulled", "Completed" }
    };

    public void ProcessRequest(HttpContext context)
    {
      if (SecurityManager.AccessAllowed())
      {
        Initialize();

        var query = HttpContext.Current.Request.QueryString;
        var request = new DataTablesPageRequest();
        request.DisplayLength = MainUtil.GetInt(query[LengthParamName], 0);
        request.DisplayStart = MainUtil.GetInt(query[StartParamName], 0);
        request.Search = GetSearch();
        request.OrderBy = GetOrderByColumn(query);
        request.OrderDir = GetOrderDir(query);

        var queue = new TranslationQueue();
        var page = queue.FindItems(request);
        var items = page.Items;
        
        var data = GetData(items.ToList());
        data.draw = MainUtil.GetInt(query[DrawParamName], 0);
        data.recordsTotal = page.TotalItems;
        data.recordsFiltered = page.TotalItems;

        var result = JsonConvert.SerializeObject(data);
        context.Response.ContentType = JsonContentType;
        context.Response.Write(result);
      }
    }

    private string GetOrderDir(NameValueCollection query)
    {
      if (query[OrderDirParamName] == "asc")
      {
        return "ASC";
      }

      return "DESC";
    }

    private string GetOrderByColumn(NameValueCollection query)
    {
      if (query[OrderByParamName] == PathColumnIndex)
      {
        return "Path";
      }

      if (query[OrderByParamName] == SubmittedColumnIndex)
      {
        return "Submitted";
      }

      return "Applied";
    }

    private void Initialize()
    {
      foreach (var language in LanguageManager.GetLanguages(Database.GetDatabase(ModuleConstants.DefaultDatabase)))
      {
        languages.Add(language.Name, language.CultureInfo.DisplayName);
      }
    }
    
    private static DataTableSearch GetSearch()
    {
      var filters = new DataTableSearch();
      var query = HttpContext.Current.Request.QueryString;

      var languages = query["languages[]"];
      var submitters = query["submitters[]"];
      var statuses = query["statuses[]"];
      var contentType = query["contentType[]"];

      var path = query["path"];
      if (!string.IsNullOrEmpty(path))
      {
        filters.Path = path;
      }

      if (!string.IsNullOrEmpty(languages))
      {
        filters.Languages = languages.Split(',').ToList();
      }

      if (!string.IsNullOrEmpty(submitters))
      {
        filters.Submitters = submitters.Split(',').ToList();
      }

      if (!string.IsNullOrEmpty(statuses))
      {
        filters.Statuses = statuses.Split(',').ToList();
      }

      if (!string.IsNullOrEmpty(contentType))
      {
        filters.Content = contentType.Split(',').ToList();
      }

      return filters;
    }

    private RootObject GetData(List<QueuedItem> items)
    {
      var dataObject = new RootObject();
      dataObject.data = new List<List<string>>();

      foreach (var item in items)
      {
        var list = new List<string>();
        list.Add(item.Id.ToString());
        list.Add(string.Format("<a href='{0}' target='_blank'>{1}</a>", item.ItemId.ContentEditorUrl(item.SourceLanguage), item.Path));
        list.Add(GetSourceLanguage(item));
        list.Add(GetTargetLanguage(item));

        var status = GetStatusTitle(item);
        list.Add(string.Format("<span title='{0}'>{1}</span>", item.LastError, status));

        if (item.MediaContent)
        {
          list.Add(Translate.Text(MediaFileType));
          list.Add(string.Format(PercentCompleteMediaTemplate, item.PercentComplete));
        }
        else
        {
          list.Add(Translate.Text(ContentFileType));
          list.Add(string.Format(PercentCompleteTemplate, item.ItemId, item.TargetLanguage, item.PercentComplete));
        }

        list.Add(item.SubmittedBy);
        list.Add(item.WordCount > 0 ? item.WordCount.ToString(CultureInfo.InvariantCulture) : string.Empty);
        list.Add(string.Format(DateFormatString, DateUtil.ToIsoDate(item.Submitted), item.Submitted.ToString(LongDateFormat), item.Submitted.ToString(ShortDateFormat)));

        if (item.Applied.HasValue)
        {
          list.Add(string.Format(DateFormatString, DateUtil.ToIsoDate(item.Applied.Value), item.Applied.Value.ToString(LongDateFormat), item.Applied.Value.ToString(ShortDateFormat)));
        }
        else
        {
          list.Add("<span></span>");
        }

        dataObject.data.Add(list);
      }

      return dataObject;
    }

    private string GetTargetLanguage(QueuedItem item)
    {
      if (languages.ContainsKey(item.TargetLanguage))
      {
        return languages[item.TargetLanguage];
      }

      return item.TargetLanguage;
    }

    private string GetSourceLanguage(QueuedItem item)
    {
      if (languages.ContainsKey(item.SourceLanguage))
      {
        return languages[item.SourceLanguage];
      }

      return item.SourceLanguage;
    }


    private static string GetStatusTitle(QueuedItem item)
    {
      if (statusTitles.ContainsKey(item.Status))
      {
        return statusTitles[item.Status];
      }

      return item.Status;
    }

    public bool IsReusable
    {
      get
      {
        return false;
      }
    }
  }
}